/*
 * pwm_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
 
/******************************************************************************
 *
 * The vPWMTask task configures the PWM0 peripheral of the TM4C123GH6PM MCU to
 * output a PWM on PB6 that runs at 250 Hz with an initial duty cycle of 25%.
 * In order to get such a low speed PWM, the PWM clock is set to be the system
 * clock divided by 8.  The LOAD interrupt is then configured for PWM0 Gen 0.
 * This interrupt will trigger every time the PWM0 counter gets reloaded. The
 * processing of this interrupt will be deferred to prvPWMIntTask.
 *
 * The prvPWMIntTask will check if the duty cycle is below 75% and if so then
 * it will increase the duty cycle by 0.1%.  Once the duty cycle reaches 75%,
 * the task will then reset the duty cycle back down to 25%.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/pwm.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
/*-----------------------------------------------------------*/

/*
 * Declare a variable that is used to hold the handle of the PWM
 * interrupt task.
 */
TaskHandle_t xPWMIntTask = NULL;

/*
 * The variable g_ui32PWMIncrement contains the value needed to increment the
 * PWM duty cycle by 0.1% cycles based on the System Clock speed.
 */
uint32_t g_ui32PWMIncrement;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvPWMIntTask( void *pvParameters );

/*
 * Called by main() to create the PWM task.
 */
void vPWMTask( void );

/* 
 * Configure the PWM 0 peripheral on the TM4C123GH6PM to output a 250 Hz PWM
 * at 25% duty cycle and enable interrupts for PWM reload.
 */
static void prvConfigurePWM( void );
/*-----------------------------------------------------------*/

void vPWMTask( void )
{
    /* Configure the PWM module. */
    prvConfigurePWM();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name PWM0 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvPWMIntTask,
                 "PWM0",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xPWMIntTask );
}

/*-----------------------------------------------------------*/

static void prvPWMIntTask( void *pvParameters )
{
unsigned long ulEventsToProcess;

    for( ;; )
	{
        /* Wait to receive a notification sent directly to this task from the
        interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* If the duty cycle is less or equal to 75% then add 0.1% to the duty
             * cycle.  Else, reset the duty cycle to 25%. */
            if((PWMPulseWidthGet(PWM0_BASE, PWM_OUT_0) + g_ui32PWMIncrement) <=
               ((PWMGenPeriodGet(PWM0_BASE, PWM_GEN_0) * 3) / 4))
            {
                PWMPulseWidthSet(PWM0_BASE, PWM_OUT_0,
                                     PWMPulseWidthGet(PWM0_BASE, PWM_GEN_0) +
                                     g_ui32PWMIncrement);
            }
            else
            {
                PWMPulseWidthSet(PWM0_BASE, PWM_OUT_0,
                                 PWMGenPeriodGet(PWM0_BASE, PWM_GEN_0) / 4);
            }
        }
        else
        {
            /* Timed out. */
        }
	}
}
/*-----------------------------------------------------------*/

static void prvConfigurePWM( void )
{
    /* Set the PWM clock to be equal to the system clock. */
    SysCtlPWMClockSet(SYSCTL_PWMDIV_4);

    /* The PWM peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_PWM0);

    /* Configure the PWM function for this pin. */
    GPIOPinConfigure(GPIO_PB6_M0PWM0);
    GPIOPinTypePWM(GPIO_PORTB_BASE, GPIO_PIN_6);

    /* Set the PWM clock to be SysClk / 8. */
    SysCtlPWMClockSet(SYSCTL_PWMDIV_8);

    /* Configure PWM0 to count up/down without synchronization. */
    PWMGenConfigure(PWM0_BASE, PWM_GEN_0, PWM_GEN_MODE_UP_DOWN |
                    PWM_GEN_MODE_NO_SYNC);

    /* Set the PWM period to 250Hz.  To calculate the appropriate parameter use
     * the following equation: N = (1 / f) * PWMClk.  Where N is the function
     * parameter, f is the desired frequency, and PWMClk is the PWM clock
     * frequency that is derived from the system clock frequency divided by the
     * PWM clock divider used (if any). */
    PWMGenPeriodSet(PWM0_BASE, PWM_GEN_0, ( (configCPU_CLOCK_HZ / 8 ) / 250) );

    /* Set the PWM increment variable based on the PWM Clock which is the
     * system clock divided by eight.  Since this is a 250 Hz PWM, continue
     * to use the equation N = (1 / f) * SysClk.  Then to set the initial
     * period to 0.1% by dividing (N / 1000).  This variable will be used
     * to increment PWM0 by 0.1% on each interrupt. */
    g_ui32PWMIncrement = (((configCPU_CLOCK_HZ / 8) / 250) / 1000);

    /* Set PWM0 to a duty cycle of 25%.  You set the duty cycle as a function
     * of the period.  Since the period was set above, you can use the
     * PWMGenPeriodGet() function. */
    PWMPulseWidthSet(PWM0_BASE, PWM_OUT_0,
                     PWMGenPeriodGet(PWM0_BASE, PWM_GEN_0) / 4);

    /* Allow PWM0 generated interrupts.  This configuration is done to
     * differentiate fault interrupts from other PWM0 related interrupts. */
    PWMIntEnable(PWM0_BASE, PWM_INT_GEN_0);

    /* Enable the PWM0 LOAD interrupt on PWM Gen 0. */
    PWMGenIntTrigEnable(PWM0_BASE, PWM_GEN_0, PWM_INT_CNT_LOAD);

    /* Enable the interrupt for PWM Gen 0 on the processor (NVIC). */
    IntEnable(INT_PWM0_0);

    /* Enable PWM Out Bit 0 (PB6) output signal. */
    PWMOutputState(PWM0_BASE, PWM_OUT_0_BIT, true);

    /* Enable the PWM generator block. */
    PWMGenEnable(PWM0_BASE, PWM_GEN_0);
}
/*-----------------------------------------------------------*/

void xPWMGen0Handler( void )
{
BaseType_t xHigherPriorityTaskWoken;

    /* Clear the hardware interrupt flag for the PWM0 LOAD interrupt.  This flag
     * gets set when the PWM counter gets reloaded. */
    PWMGenIntClear(PWM0_BASE, PWM_GEN_0, PWM_INT_CNT_LOAD);

    /* The xHigherPriorityTaskWoken parameter must be initialized to pdFALSE as
    it will get set to pdTRUE inside the interrupt safe API function if a
    context switch is required. */
    xHigherPriorityTaskWoken = pdFALSE;

    /* Defer the interrupt processing to a Task to minimize time spent within
     * the hardware interrupt service routine.  Send a notification directly to
     * the task to which interrupt processing is being deferred. */
    vTaskNotifyGiveFromISR( xPWMIntTask, &xHigherPriorityTaskWoken );

    /* This FreeRTOS API call will handle the context switch if it is required
     * or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xHigherPriorityTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
		configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
		added here, but the tick hook is called from an interrupt context, so
		code must not attempt to block, and only the interrupt safe FreeRTOS API
		functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
		executed here. */
}


